package net.aocat.psis.client.samples.verify.signature;

import java.io.File;
import java.io.IOException;
import java.util.List;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.CertificateAttributes;
import net.aocat.psis.client.utils.CertificateSerialConverter;
import net.aocat.psis.client.utils.Profiles;
import net.aocat.psis.client.utils.SignatureAttributes;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;

import x0Assertion.oasisNamesTcSAML2.AttributeType;
import x0CoreSchema.oasisNamesTcDss1.SignatureObjectType;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.OptionalOutputsDocument.OptionalOutputs;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument.VerifyRequest;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnSignatureInfoDocument.ReturnSignatureInfo;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnX509CertificateInfoDocument.ReturnX509CertificateInfo;

/**
 * Java code sample for XAdES enveloping signature validation.
 * @author aalcaide
 */
public class XAdESSignatureEnvelopingValidationSample extends AbstractSample {

	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String VERIFY_REQUEST = "XAdES-Enveloping-VerifyRequest.xml";
	private String VERIFY_RESPONSE = "XAdES-Enveloping-VerifyResponse.xml";

	public XAdESSignatureEnvelopingValidationSample(){
		super();
	}

	/**
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args){
		try{
			File signature = new File("resources\\signatures\\xml\\XAdES-T-enveloping.xml");
			XAdESSignatureEnvelopingValidationSample validation = new XAdESSignatureEnvelopingValidationSample();
			validation.verifyXAdESSignatureEnveloping(signature);
		} catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Verifies a XAdES enveloping signature.
	 * Prints and/or saves request and response. Prints certificate and signature attributes.
	 * Saves updated signature to disk.
	 * @param signature signature to validate
	 * @throws IOException
	 * @throws XmlException
	 */
	public void verifyXAdESSignatureEnveloping(File signature) throws IOException, XmlException {

		this.log.info("Start validation "+Profiles.XSS);

		this.log.debug("Start reading File "+signature.getAbsolutePath());

		this.log.debug("Start message creation ");

		//VerifyRequest
		VerifyRequestDocument verifyRequest = VerifyRequestDocument.Factory.newInstance();
		VerifyRequest request = verifyRequest.addNewVerifyRequest();
		//XSS Profile
		request.setProfile(Profiles.XSS);

		//signature
		SignatureObjectType signObj = request.addNewSignatureObject();
		signObj.set(XmlObject.Factory.parse(signature));

		//optional inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();
		
		//processing details
		optInputs.addNewReturnProcessingDetails();
		
		//certificate attributes
		ReturnX509CertificateInfo certInfo = optInputs.addNewReturnX509CertificateInfo();
		List<String> attributesNames = CertificateAttributes.getSomeImportantAttributes();
		for(String attName:attributesNames){
			AttributeType att = certInfo.addNewAttributeDesignator();
			att.setName(attName);
		}
		
		//signature attributes
		ReturnSignatureInfo signInfo = optInputs.addNewReturnSignatureInfo();
		AttributeType signAtt = signInfo.addNewAttributeDesignator();
		signAtt.setName(SignatureAttributes.ExpirationDate);
		
		//print and save request
		printSaveRequest(PRINT_REQUEST, verifyRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_REQUEST);

		//send request to PSIS
		this.log.debug("Send signature validation request");
		//VerifyResponse
		VerifyResponseDocument verifyResponse = this.factory.getDssPort().verify(verifyRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, verifyResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_RESPONSE);
		
		//optional outputs
		OptionalOutputs optOutputs = verifyResponse.getVerifyResponse().getOptionalOutputs();
		
		//retrieving certificate attribute values
		this.log.debug("Certificate attribute values:");
		List<AttributeType> attValues = optOutputs.getX509CertificateInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName();
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//serial: convert decimal value to hexadecimal
				if(name.equalsIgnoreCase(CertificateAttributes.SerialNumber))
					value = CertificateSerialConverter.convertDecSerial2Hex(value);
				this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving signature attributes
		attValues = optOutputs.getSignatureInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName(); 
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//expiration date
				if(name.equalsIgnoreCase(SignatureAttributes.ExpirationDate))
					this.log.debug("Signature expiration date: "+attValue.getAttributeValueArray(0).getDomNode().getFirstChild().getNodeValue());
				else
					this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving updated signature
		SignatureObjectType updatedSignature = optOutputs.getUpdatedSignatureArray(0).getSignatureObject();
		this.log.debug("Saving updated signature to "+signature+"_updated.xml");
		updatedSignature.save(new File(signature+"_updated.xml"));

	}

}
