package net.aocat.psis.client.samples.verify.signature;

import java.io.File;
import java.io.IOException;
import java.util.List;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.CertificateAttributes;
import net.aocat.psis.client.utils.CertificateSerialConverter;
import net.aocat.psis.client.utils.Profiles;
import net.aocat.psis.client.utils.SignatureAttributes;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;

import x0Assertion.oasisNamesTcSAML2.AttributeType;
import x0CoreSchema.oasisNamesTcDss1.DocumentType;
import x0CoreSchema.oasisNamesTcDss1.DocumentWithSignatureDocument.DocumentWithSignature;
import x0CoreSchema.oasisNamesTcDss1.InlineXMLType;
import x0CoreSchema.oasisNamesTcDss1.InputDocumentsDocument.InputDocuments;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.OptionalOutputsDocument.OptionalOutputs;
import x0CoreSchema.oasisNamesTcDss1.SignatureObjectType;
import x0CoreSchema.oasisNamesTcDss1.SignaturePtrDocument.SignaturePtr;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument.VerifyRequest;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnSignatureInfoDocument.ReturnSignatureInfo;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnX509CertificateInfoDocument.ReturnX509CertificateInfo;

/**
 * Java code sample for XAdES enveloped signature validation.
 * @author aalcaide
 */
public class XAdESSignatureEnvelopedValidationSample extends AbstractSample {

	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String VERIFY_REQUEST = "XAdES-Enveloped-VerifyRequest.xml";
	private String VERIFY_RESPONSE = "XAdES-Enveloped-VerifyResponse.xml";

	public XAdESSignatureEnvelopedValidationSample(){
		super();
	}

	/**
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args){
		try{
			File signature = new File("resources\\signatures\\xml\\XAdES-T-enveloped.xml");
			XAdESSignatureEnvelopedValidationSample validation = new XAdESSignatureEnvelopedValidationSample();
			validation.verifyXAdESSignatureEnveloped(signature);
		} catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Verifies a XAdES enveloped signature.
	 * Prints and/or saves request and response. Prints certificate and signature attributes.
	 * Saves updated signature to disk.
	 * @param docWithSignature document with signature to validate
	 * @throws IOException
	 * @throws XmlException
	 */
	public void verifyXAdESSignatureEnveloped(File docWithSignature) throws IOException, XmlException {

		this.log.info("Start validation "+Profiles.XSS);

		this.log.debug("Start reading File "+docWithSignature.getAbsolutePath());

		this.log.debug("Start message creation ");

		//VerifyRequest
		VerifyRequestDocument verifyRequest = VerifyRequestDocument.Factory.newInstance();
		VerifyRequest request = verifyRequest.addNewVerifyRequest();
		//XSS Profile
		request.setProfile(Profiles.XSS);

		//signature
		//document with signature
		InputDocuments inDocs = request.addNewInputDocuments();
		DocumentType doc = inDocs.addNewDocument();
		doc.setID("docId");
		doc.setRefURI("");
		InlineXMLType inXml = doc.addNewInlineXML();
		inXml.set(XmlObject.Factory.parse(docWithSignature));
		//signature pointer
		SignatureObjectType signObj = request.addNewSignatureObject();
		SignaturePtr signPtr = signObj.addNewSignaturePtr();
		signPtr.setWhichDocument("docId");

		//optional inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();
		
		//processing details
		optInputs.addNewReturnProcessingDetails();
		
		//certificate attributes
		ReturnX509CertificateInfo certInfo = optInputs.addNewReturnX509CertificateInfo();
		List<String> attributesNames = CertificateAttributes.getSomeImportantAttributes();
		for(String attName:attributesNames){
			AttributeType att = certInfo.addNewAttributeDesignator();
			att.setName(attName);
		}
		
		//signature attributes
		ReturnSignatureInfo signInfo = optInputs.addNewReturnSignatureInfo();
		AttributeType signAtt = signInfo.addNewAttributeDesignator();
		signAtt.setName(SignatureAttributes.ExpirationDate);

		//print and save request
		printSaveRequest(PRINT_REQUEST, verifyRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_REQUEST);

		//send request to PSIS
		this.log.debug("Send signature validation request");
		//VerifyResponse
		VerifyResponseDocument verifyResponse = this.factory.getDssPort().verify(verifyRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, verifyResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_RESPONSE);

		//optional outputs
		OptionalOutputs optOutputs = verifyResponse.getVerifyResponse().getOptionalOutputs();
		
		//retrieving certificate attribute values
		this.log.debug("Certificate attribute values:");
		List<AttributeType> attValues = verifyResponse.getVerifyResponse().getOptionalOutputs().getX509CertificateInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName();
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//serial: convert decimal value to hexadecimal
				if(name.equalsIgnoreCase(CertificateAttributes.SerialNumber))
					value = CertificateSerialConverter.convertDecSerial2Hex(value);
				this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving signature attributes
		attValues = optOutputs.getSignatureInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName(); 
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//expiration date
				if(name.equalsIgnoreCase(SignatureAttributes.ExpirationDate))
					this.log.debug("Signature expiration date: "+attValue.getAttributeValueArray(0).getDomNode().getFirstChild().getNodeValue());
				else
					this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving updated signature
		String docId = optOutputs.getUpdatedSignatureArray(0).getSignatureObject().getSignaturePtr().getWhichDocument();
		List<DocumentWithSignature> outDocs = optOutputs.getDocumentWithSignatureList();
		for(DocumentWithSignature outDoc:outDocs){
			//if there is a document with the pointed ID, we retrieve it
			if(outDoc.getDocument().getID().equalsIgnoreCase(docId)){
				//because of XmlBeans behavior regarding namespaces, we retrieve the signature with a query from the complete response 
				//(for it not to be broken with added namespaces on top)
				//VERY IMPORTANT: In case you are not sure of saving correctly the document with signature, save the PSIS complete response, as it remains unbroken within. 
				//Or validate the updated signature to PSIS to be sure it's not broken.
				String query = "declare namespace dss='urn:oasis:names:tc:dss:1.0:core:schema'; $this/dss:VerifyResponse/dss:OptionalOutputs/dss:DocumentWithSignature/dss:Document[@ID='"+docId+"']/dss:InlineXML/*";
				XmlObject[] docsWithUpdatedSignature = verifyResponse.execQuery(query);
				for(XmlObject docWithUpdatedSignature:docsWithUpdatedSignature){
					this.log.debug("Saving updated signature to "+docWithSignature+"_updated.xml");
					docWithUpdatedSignature.save(new File(docWithSignature+"_updated.xml"));
				}
			}	
		}

	}

}
