package net.aocat.psis.client.samples.verify.signature;

import java.io.File;
import java.io.IOException;
import java.util.List;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.CertificateAttributes;
import net.aocat.psis.client.utils.CertificateSerialConverter;
import net.aocat.psis.client.utils.Profiles;
import net.aocat.psis.client.utils.SignatureAttributes;
import net.aocat.psis.client.utils.Types;
import net.aocat.psis.client.utils.Utils;

import org.apache.commons.io.FileUtils;
import org.apache.xmlbeans.XmlObject;

import x0Assertion.oasisNamesTcSAML2.AttributeType;
import x0CoreSchema.oasisNamesTcDss1.Base64SignatureDocument.Base64Signature;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.OptionalOutputsDocument.OptionalOutputs;
import x0CoreSchema.oasisNamesTcDss1.SignatureObjectType;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument.VerifyRequest;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnSignatureInfoDocument.ReturnSignatureInfo;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnX509CertificateInfoDocument.ReturnX509CertificateInfo;

/**
 * Java code sample for CAdES attached signature validation.
 * @author aalcaide
 */
public class CAdESSignatureAttachedValidationSample extends AbstractSample {

	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String VERIFY_REQUEST = "CAdES-Attached-VerifyRequest.xml";
	private String VERIFY_RESPONSE = "CAdES-Attached-VerifyResponse.xml";

	public CAdESSignatureAttachedValidationSample(){
		super();
	}

	/**
	 * @param args
	 */
	public static void main(String[] args){
		try{
			File signature = new File("resources\\signatures\\cms\\CAdES-T-attached.p7s");
			CAdESSignatureAttachedValidationSample validation = new CAdESSignatureAttachedValidationSample();
			validation.verifyCAdESSignatureAttached(signature);
		} catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Verifies a CAdES attached signature.
	 * Prints and/or saves request and response. Prints certificate and signature attributes.
	 * Saves updated signature to disk.
	 * @param signature signature to validate and update
	 * @throws IOException
	 */
	public void verifyCAdESSignatureAttached(File signature) throws IOException {

		this.log.info("Start validation "+Profiles.XSS);

		this.log.debug("Start reading File "+signature.getAbsolutePath());
		byte[] signBytes = FileUtils.readFileToByteArray(signature);

		this.log.debug("Start message creation ");

		//VerifyRequest
		VerifyRequestDocument verifyRequest = VerifyRequestDocument.Factory.newInstance();
		VerifyRequest request = verifyRequest.addNewVerifyRequest();
		//XSS Profile
		request.setProfile(Profiles.XSS);

		//signature
		SignatureObjectType signObj = request.addNewSignatureObject();
		Base64Signature b64Sign = signObj.addNewBase64Signature();
		b64Sign.setType(Types.Signature.CAdES);
		b64Sign.setByteArrayValue(signBytes);

		//optional inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();
		
		//processing details
		optInputs.addNewReturnProcessingDetails();
		
		//certificate attributes
		ReturnX509CertificateInfo certInfo = optInputs.addNewReturnX509CertificateInfo();
		List<String> attributesNames = CertificateAttributes.getSomeImportantAttributes();
		for(String attName:attributesNames){
			AttributeType att = certInfo.addNewAttributeDesignator();
			att.setName(attName);
		}
		
		//signature attributes
		ReturnSignatureInfo signInfo = optInputs.addNewReturnSignatureInfo();
		AttributeType signAtt = signInfo.addNewAttributeDesignator();
		signAtt.setName(SignatureAttributes.ExpirationDate);

		//print and save request
		printSaveRequest(PRINT_REQUEST, verifyRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_REQUEST);

		//send request to PSIS
		this.log.debug("Send signature validation request");
		//VerifyResponse
		VerifyResponseDocument verifyResponse = this.factory.getDssPort().verify(verifyRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, verifyResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_RESPONSE);

		//optional outputs
		OptionalOutputs optOutputs = verifyResponse.getVerifyResponse().getOptionalOutputs();
		
		//retrieving certificate attribute values
		this.log.debug("Certificate attribute values:");
		List<AttributeType> attValues = verifyResponse.getVerifyResponse().getOptionalOutputs().getX509CertificateInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName();
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//serial: convert decimal value to hexadecimal
				if(name.equalsIgnoreCase(CertificateAttributes.SerialNumber))
					value = CertificateSerialConverter.convertDecSerial2Hex(value);
				this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving signature attributes
		attValues = optOutputs.getSignatureInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName(); 
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//expiration date
				if(name.equalsIgnoreCase(SignatureAttributes.ExpirationDate))
					this.log.debug("Signature expiration date: "+attValue.getAttributeValueArray(0).getDomNode().getFirstChild().getNodeValue());
				else
					this.log.debug(name+" --> "+value);				
			}
		}
		
		//retrieving updated signature
		byte[] updatedSignature = optOutputs.getUpdatedSignatureArray(0).getSignatureObject().getBase64Signature().getByteArrayValue();
		this.log.debug("Saving updated signature to "+signature+"_updated.p7s");
		Utils.saveToFile(updatedSignature, signature+"_updated.p7s");
		
	}

}
