package net.aocat.psis.client.samples.verify.pdf;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import javax.swing.text.DateFormatter;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.Forms;
import net.aocat.psis.client.utils.PDFSignatureAttributes;
import net.aocat.psis.client.utils.Profiles;
import net.aocat.psis.client.utils.SignatureAttributes;
import net.aocat.psis.client.utils.Utils;

import org.apache.commons.io.FileUtils;
import org.apache.xmlbeans.XmlObject;

import x0Assertion.oasisNamesTcSAML2.AttributeType;
import x0CoreSchema.oasisNamesTcDss1.Base64DataDocument.Base64Data;
import x0CoreSchema.oasisNamesTcDss1.DocumentWithSignatureDocument.DocumentWithSignature;
import x0CoreSchema.oasisNamesTcDss1.InputDocumentsDocument.InputDocuments;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.OptionalOutputsDocument.OptionalOutputs;
import x0CoreSchema.oasisNamesTcDss1.ReturnUpdatedSignatureDocument.ReturnUpdatedSignature;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument.VerifyRequest;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;
import x0ProfilesDSSPDF.oasisNamesTcDss1.PDFSignatureDetailsDocument.PDFSignatureDetails;
import x0ProfilesDSSPDF.oasisNamesTcDss1.PDFSignatureInfoDocument.PDFSignatureInfo;
import x0ProfilesDSSPDF.oasisNamesTcDss1.ReturnPDFSignatureInfoDocument.ReturnPDFSignatureInfo;
import x0ProfilesDSSPDF.oasisNamesTcDss1.ReturnPDFSignaturesDetailsDocument.ReturnPDFSignaturesDetails;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnSignatureInfoDocument.ReturnSignatureInfo;
import x0ProfilesXSS.oasisNamesTcDss1.SignatureInfoDocument.SignatureInfo;

/**
 * Java code sample for signed PDF validation and update.
 * @author aalcaide
 */
public class PdfValidationSample extends AbstractSample {

	private static final String MIME_TYPE_APPLICATION_PDF = "application/pdf";
	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String VERIFY_REQUEST = "PDF-VerifyRequest.xml";
	private String VERIFY_RESPONSE = "PDF-VerifyResponse.xml";
	private boolean SAVE_UPDATED_PDF = true;

	public PdfValidationSample(){
		super();
	}

	/**
	 * @param args
	 */
	public static void main(String [] args){
		try{
			File pdf = new File("resources\\signatures\\pdf\\prova.pdf");			
			PdfValidationSample validation = new PdfValidationSample();
			validation.verifyPdf(pdf);
		}catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Verifies a signed PDF, and updates it to LTV. 
	 * Prints and/or saves request and response. Prints signature/s attribute/s.
	 * Saves updated PDF to disk.
	 * @param pdf signed PDF to validate and update
	 * @throws IOException
	 * @throws ParseException
	 */
	public void verifyPdf(File pdf) throws IOException, ParseException{

		this.log.info("Start validation "+Profiles.PDF);

		this.log.debug("Start reading File "+pdf.getAbsoluteFile());

		byte[] bytes = FileUtils.readFileToByteArray(pdf);

		this.log.debug("Start message creation ");

		//VerifyRequest
		VerifyRequestDocument verifyRequest = VerifyRequestDocument.Factory.newInstance();
		VerifyRequest request = verifyRequest.addNewVerifyRequest();
		//PDF Profile
		request.setProfile(Profiles.PDF);

		//Input Documents
		InputDocuments inputDocument = request.addNewInputDocuments();
		Base64Data base64Data = Base64Data.Factory.newInstance();
		base64Data.setByteArrayValue(bytes);
		base64Data.setMimeType(MIME_TYPE_APPLICATION_PDF);
		inputDocument.addNewDocument().setBase64Data(base64Data);

		//Optional Inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();
		
		//Processing Details
		optInputs.addNewReturnProcessingDetails();

		//Return Total Revisions Number
		optInputs.addNewReturnPDFTotalRevisionsNumber();

		//Return PDF Expiration Date Before Update
		optInputs.addNewReturnPDFExpirationDateBeforeUpdate();
		
		//Return PDF Expiration Date
		optInputs.addNewReturnPDFExpirationDate();

		//Return PDF signatures details
		ReturnPDFSignaturesDetails signsDetails = optInputs.addNewReturnPDFSignaturesDetails();
		//signature info
		ReturnSignatureInfo signInfo = signsDetails.addNewReturnSignatureInfo();
		AttributeType att = signInfo.addNewAttributeDesignator();
		att.setName(SignatureAttributes.ExpirationDate);		
		//PDf signature info
		ReturnPDFSignatureInfo pdfSignInfo = signsDetails.addNewReturnPDFSignatureInfo();
		att = pdfSignInfo.addNewAttributeDesignator();
		att.setName(PDFSignatureAttributes.Reason);
		att = pdfSignInfo.addNewAttributeDesignator();
		att.setName(PDFSignatureAttributes.Location);
		att = pdfSignInfo.addNewAttributeDesignator();
		att.setName(PDFSignatureAttributes.SignerName);
		att = pdfSignInfo.addNewAttributeDesignator();
		att.setName(PDFSignatureAttributes.ContactInfo);
		
		//Update signature to LTV
		ReturnUpdatedSignature rus = optInputs.addNewReturnUpdatedSignature();
		rus.setType(Forms.LTV);
		SAVE_UPDATED_PDF = true;

		//print and save request
		printSaveRequest(PRINT_REQUEST, verifyRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_REQUEST);

		//send request to PSIS
		this.log.debug("Send request PDF verification");
		//VerifyResponse
		VerifyResponseDocument verifyResponse = this.factory.getDssPdfPort().verify(verifyRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, verifyResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_RESPONSE);

		//optional outputs
		OptionalOutputs optOutputs = verifyResponse.getVerifyResponse().getOptionalOutputs();
		
		//retrieving signature info
		List<PDFSignatureDetails> signaturesDetails = optOutputs.getPDFSignaturesDetailsArray(0).getPDFSignatureDetailsList();
		for(PDFSignatureDetails signatureDetails:signaturesDetails){
			PDFSignatureInfo PdfSignatureInfo = signatureDetails.getPDFSignatureInfo();
			//PDF signature info
			List<AttributeType> attValues = PdfSignatureInfo.getAttributeList();
			for(AttributeType attValue:attValues){
				String name = attValue.getName();
				List<XmlObject> objValueList = attValue.getAttributeValueList();
				for(XmlObject objValue:objValueList){
					String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
					this.log.debug(name+" --> "+value);				
				}
			}
			//signature info
			SignatureInfo signatureInfo = signatureDetails.getSignatureInfo();
			attValues = signatureInfo.getAttributeList();
			for(AttributeType attValue:attValues){
				String name = attValue.getName();
				List<XmlObject> objValueList = attValue.getAttributeValueList();
				for(XmlObject objValue:objValueList){
					String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
					//expiration date
					if(name.equalsIgnoreCase(SignatureAttributes.ExpirationDate))
						this.log.debug("Signature expiration date: "+attValue.getAttributeValueArray(0).getDomNode().getFirstChild().getNodeValue());
					else
						this.log.debug(name+" --> "+value);				
				}
			}
		}

		//date format for expiration dates
		DateFormat format = new SimpleDateFormat("yyyy-mm-dd hh:mm:ss z");
		format.setTimeZone(TimeZone.getTimeZone("GMT"));
		DateFormatter formatter = new DateFormatter(format);

		//retrieving PDF expiration date before update
		Date pdfExpirationDateBeforeUpdate = optOutputs.getPDFExpirationDateBeforeUpdateArray(0).getTime();
		this.log.debug("PDF expiration date before update: "+formatter.valueToString(pdfExpirationDateBeforeUpdate));
		
		//retrieving PDF expiration date
		Date pdfExpirationDate = optOutputs.getPDFExpirationDateArray(0).getTime();
		this.log.debug("PDF expiration date: "+formatter.valueToString(pdfExpirationDate));
		
		//retrieving updated PDF
		extractUpdatedPDFfromVerifyResponse(verifyResponse, pdf);

	}

	/**
	 * Extracts updated PDF from PSIS response.
	 * @param verifyResponse PSIS verify response
	 * @param pdf original PDF (updated one is saved in the same place than original one with the "_updated.xml" suffix)
	 * @throws IOException
	 */
	private void extractUpdatedPDFfromVerifyResponse(VerifyResponseDocument verifyResponse, File pdf) throws IOException{
		if(SAVE_UPDATED_PDF){
			if(verifyResponse.getVerifyResponse().getOptionalOutputs()!=null
					&& verifyResponse.getVerifyResponse().getOptionalOutputs().getDocumentWithSignatureArray(0)!=null){
				this.log.debug("Saving updated PDF to "+pdf+"_updated.pdf");
				DocumentWithSignature dws = verifyResponse.getVerifyResponse().getOptionalOutputs().getDocumentWithSignatureArray(0);
				Base64Data b64Data = dws.getDocument().getBase64Data();
				Utils.saveToFile(b64Data.getByteArrayValue(), pdf+"_updated.pdf");
			}
		}
	}

}
