package net.aocat.psis.client.samples.verify.certificate;

import java.io.File;
import java.io.IOException;
import java.util.List;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.CertificateAttributes;
import net.aocat.psis.client.utils.CertificateSerialConverter;
import net.aocat.psis.client.utils.Profiles;

import org.apache.commons.io.FileUtils;
import org.apache.xmlbeans.XmlObject;
import org.w3.x2000.x09.xmldsig.X509DataDocument;
import org.w3.x2000.x09.xmldsig.X509DataType;

import x0Assertion.oasisNamesTcSAML2.AttributeType;
import x0CoreSchema.oasisNamesTcDss1.SignatureObjectType;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.OptionalOutputsDocument.OptionalOutputs;
import x0CoreSchema.oasisNamesTcDss1.SignatureObjectType.Other;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument.VerifyRequest;
import x0ProfilesXSS.oasisNamesTcDss1.ReturnX509CertificateInfoDocument.ReturnX509CertificateInfo;

/**
 * Java code sample for certificate validation.
 * @author aalcaide
 */
public class CertificateValidationSample extends AbstractSample {

	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String VERIFY_REQUEST = "Cert-VerifyRequest.xml";
	private String VERIFY_RESPONSE = "Cert-VerifyResponse.xml";

	public CertificateValidationSample(){
		super();
	}

	/**
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args){
		try{
			File cert = new File("resources\\certificates\\20140509_090153_CPISR-1_C_cpisrc_c1_policy.crt");
			CertificateValidationSample validation = new CertificateValidationSample();
			validation.verifyCert(cert);
		} catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Verifies a certificate. Prints and/or saves request and response. Prints certificate attributes.
	 * @param cert certificate to validate
	 * @throws IOException
	 */
	public void verifyCert(File cert) throws IOException {

		this.log.info("Start validation "+Profiles.XSS);

		this.log.debug("Start reading File "+cert.getAbsolutePath());
		byte[] certBytes = FileUtils.readFileToByteArray(cert);

		this.log.debug("Start message creation ");

		//VerifyRequest
		VerifyRequestDocument verifyRequest = VerifyRequestDocument.Factory.newInstance();
		VerifyRequest request = verifyRequest.addNewVerifyRequest();
		//XSS Profile
		request.setProfile(Profiles.XSS);

		//certificate
		X509DataDocument x509Doc = X509DataDocument.Factory.newInstance();
		X509DataType x509Data = x509Doc.addNewX509Data();
		x509Data.addX509Certificate(certBytes);
		SignatureObjectType signatureObject = request.addNewSignatureObject();
		Other any = signatureObject.addNewOther();
		any.set(x509Doc);

		//Optional Inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();
		
		//processing details
		optInputs.addNewReturnProcessingDetails();
		
		//certificate attributes
		ReturnX509CertificateInfo certInfo = optInputs.addNewReturnX509CertificateInfo();
		List<String> attributesNames = CertificateAttributes.getSomeImportantAttributes();
		for(String attName:attributesNames){
			AttributeType att = certInfo.addNewAttributeDesignator();
			att.setName(attName);
		}
		
		//print and save request
		printSaveRequest(PRINT_REQUEST, verifyRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_REQUEST);

		//send request to PSIS
		this.log.debug("Send certificate validation request");
		//VerifyResponse
		VerifyResponseDocument verifyResponse = this.factory.getDssPort().verify(verifyRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, verifyResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, VERIFY_RESPONSE);
		
		//optional outputs
		OptionalOutputs optOutputs = verifyResponse.getVerifyResponse().getOptionalOutputs();
		
		//retrieving certificate attribute values
		this.log.debug("Certificate attribute values:");
		List<AttributeType> attValues = optOutputs.getX509CertificateInfoArray(0).getAttributeList();
		for(AttributeType attValue:attValues){
			String name = attValue.getName();
			List<XmlObject> objValueList = attValue.getAttributeValueList();
			for(XmlObject objValue:objValueList){
				String value = objValue.getDomNode().getFirstChild().getNodeValue(); 
				//serial: convert decimal value to hexadecimal
				if(name.equalsIgnoreCase(CertificateAttributes.SerialNumber))
					value = CertificateSerialConverter.convertDecSerial2Hex(value);
				this.log.debug(name+" --> "+value);				
			}
		}

	}

}
