package net.aocat.psis.client.samples.timestamp.create;

import java.io.File;
import java.io.IOException;

import net.aocat.psis.client.base.AbstractSample;
import net.aocat.psis.client.utils.Profiles;
import net.aocat.psis.client.utils.Types;

import org.apache.commons.io.FileUtils;
import org.apache.xmlbeans.XmlBase64Binary;
import org.w3.x2000.x09.xmldsig.DigestMethodType;
import org.w3.x2000.x09.xmldsig.KeyInfoType;
import org.w3.x2000.x09.xmldsig.X509DataType;

import x0CoreSchema.oasisNamesTcDss1.SignRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.SignResponseDocument;
import x0CoreSchema.oasisNamesTcDss1.DocumentHashDocument.DocumentHash;
import x0CoreSchema.oasisNamesTcDss1.InputDocumentsDocument.InputDocuments;
import x0CoreSchema.oasisNamesTcDss1.KeySelectorDocument.KeySelector;
import x0CoreSchema.oasisNamesTcDss1.OptionalInputsDocument.OptionalInputs;
import x0CoreSchema.oasisNamesTcDss1.SignRequestDocument.SignRequest;

/**
 * Java code sample for XML timestamp creation.
 * @author aalcaide
 */
public class XMLTimestampCreationSample extends AbstractSample {

	private boolean PRINT_REQUEST = true;
	private boolean PRINT_RESPONSE = true;
	private String OUT_DIR = "out\\";
	private boolean SAVE_REQUEST_AND_RESPONSE = true;
	private String SIGN_REQUEST = "XML-Timestamp-SignRequest.xml";
	private String SIGN_RESPONSE = "XML-Timestamp-SignResponse.xml";

	public XMLTimestampCreationSample(){
		super();
	}

	/**
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args){
		try{
			File tsaCert = new File("resources\\timestamp\\20141105_Servei_de_segellat_de_temps_de_PSIS_2014-11-05.crt");
			File data = new File("resources\\timestamp\\xml\\hash.bin");
			XMLTimestampCreationSample creation = new XMLTimestampCreationSample();
			creation.createTimestamp(data, tsaCert);
		} catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * Creates an XML timestamp and saves it to disc. Prints and/or saves request and response.
	 * @param dataHash data hash to timestamp
	 * @param tsaCert TSA certificate
	 * @throws IOException
	 */
	public void createTimestamp(File dataHash, File tsaCert) throws IOException {

		this.log.info("Start creation "+Profiles.TIMESTAMPING);

		this.log.debug("Start reading File "+dataHash.getAbsolutePath());
		byte[] dataHashBytes = FileUtils.readFileToByteArray(dataHash);

		this.log.debug("Start message creation ");

		//SignRequest
		SignRequestDocument signRequest = SignRequestDocument.Factory.newInstance();
		SignRequest request = signRequest.addNewSignRequest();
		//TIMESTAMPING Profile
		request.setProfile(Profiles.TIMESTAMPING);

		//input documents
		InputDocuments inDocs = request.addNewInputDocuments();
		DocumentHash doc = inDocs.addNewDocumentHash();
		DigestMethodType digMethod = doc.addNewDigestMethod();
		digMethod.setAlgorithm("http://www.w3.org/2000/09/xmldsig#sha1");
		doc.setDigestValue(dataHashBytes);

		//optional inputs
		OptionalInputs optInputs = request.addNewOptionalInputs();

		//signature type -> XML timestamp
		optInputs.addSignatureType(Types.TimeStamp.XML);

		//key selector
		KeySelector keySel = optInputs.addNewKeySelector();
		KeyInfoType keyInfo = keySel.addNewKeyInfo();
		X509DataType x509Data = keyInfo.addNewX509Data();
		XmlBase64Binary cert = x509Data.addNewX509Certificate();
		cert.setByteArrayValue(FileUtils.readFileToByteArray(tsaCert));

		//print and save request
		printSaveRequest(PRINT_REQUEST, signRequest, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, SIGN_REQUEST);

		//send request to PSIS
		this.log.debug("Send timestamp creation request");
		//SignResponse
		SignResponseDocument signResponse = this.factory.getDssPort().sign(signRequest);

		//print and save response
		printSaveResponse(PRINT_RESPONSE, signResponse, SAVE_REQUEST_AND_RESPONSE, OUT_DIR, SIGN_RESPONSE);
		
		//retrieving timestamp
		this.log.debug("Saving created timestamp to "+OUT_DIR+"XMLTimestamp-created.xml");
		signResponse.getSignResponse().getSignatureObject().getTimestamp().save(new File(OUT_DIR+"XMLTimestamp-created.xml"));

	}

}
