package net.aocat.psis.client.base;

import java.io.File;
import java.io.IOException;

import net.aocat.psis.client.config.PsisClientFactoryConfigurator;
import net.aocat.psis.client.utils.Utils;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Logger;

import x0CoreSchema.oasisNamesTcDss1.ResultDocument.Result;
import x0CoreSchema.oasisNamesTcDss1.SignRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.SignResponseDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyRequestDocument;
import x0CoreSchema.oasisNamesTcDss1.VerifyResponseDocument;

/**
 * Abstract sample.
 * @author aalcaide
 */
public class AbstractSample {

	protected Logger log;

	protected PsisClientFactory factory;

	/**
	 * Init sample.
	 * Loads configuration for log4java and PSIS client.
	 */
	public AbstractSample(){
		BasicConfigurator.configure();
		this.log = Logger.getLogger(AbstractSample.class);
		this.factory = new PsisClientFactory();
		this.factory.setConfigurator(new PsisClientFactoryConfigurator());
	}

	/**
	 * Validates if PSIS verify response has a Result field or not.
	 * @param response verify response from PSIS
	 * @return true if PSIS verif yresponse has a Result field, false otherwise
	 */
	protected boolean isResultWellFormed(VerifyResponseDocument response){
		boolean ok = response!=null && response.getVerifyResponse()!=null && response.getVerifyResponse().getResult()!=null;
		return ok;
	}
	
	/**
	 * Validates if PSIS sign response has a Result field or not.
	 * @param response sign response from PSIS
	 * @return true if PSIS sign response has a Result field, false otherwise
	 */
	protected boolean isResultWellFormed(SignResponseDocument response){
		boolean ok = response!=null && response.getSignResponse()!=null && response.getSignResponse().getResult()!=null;
		return ok;
	}

	/**
	 * Prints PSIS response's Result field.
	 * @param result PSIS response's result field
	 */
	protected void showResult(Result result){
		String major = result.getResultMajor();
		String minor = result.getResultMinor();
		this.log.info("\nMajor: "+major+"\nMinor: "+minor);
		String message = null;
		if(result.getResultMessage()!=null){
			message = result.getResultMessage().getStringValue();
			this.log.info("\nMessage: "+message);
		}

	}

	/**
	 * Logs and saves PSIS verify request to a file.
	 * @param PRINT_REQUEST true if print request required
	 * @param request PSIS verify request
	 * @param SAVE_REQUEST true if save request required
	 * @param OUT_DIR directory where to save the request
	 * @param VERIFY_REQUEST name of the file where to save the request 
	 * @throws IOException
	 */
	protected void printSaveRequest(boolean PRINT_REQUEST, VerifyRequestDocument request,
			boolean SAVE_REQUEST, String OUT_DIR, String VERIFY_REQUEST) throws IOException{
		//printing request
		if (PRINT_REQUEST){
			this.log.debug("Verify request:\n");
			Utils.printXmlObject(request);
		}
		//save request to file
		if (SAVE_REQUEST==true){
			this.log.debug("Saving request to "+OUT_DIR+VERIFY_REQUEST);
			request.save(new File(OUT_DIR+VERIFY_REQUEST));
		}
	}

	/**
	 * Logs and saves PSIS verify response to a file.
	 * @param PRINT_RESPONSE true if print response required
	 * @param response PSIS verify response
	 * @param SAVE_RESPONSE true if save response required
	 * @param OUT_DIR directory where to save the response
	 * @param VERIFY_RESPONSE name of the file where to save the response
	 * @throws IOException
	 */
	protected void printSaveResponse(boolean PRINT_RESPONSE, VerifyResponseDocument response,
			boolean SAVE_RESPONSE, String OUT_DIR, String VERIFY_RESPONSE) throws IOException{
		if(isResultWellFormed(response))
			showResult(response.getVerifyResponse().getResult());
		//printing response
		if (PRINT_RESPONSE){
			this.log.debug("Verify response:\n");
			Utils.printXmlObject(response);
		}
		//save response to file
		if (SAVE_RESPONSE==true){
			this.log.debug("Saving response to "+OUT_DIR+VERIFY_RESPONSE);
			response.save(new File(OUT_DIR+VERIFY_RESPONSE));
		}
	}

	/**
	 * Logs and saves PSIS sign request to a file.
	 * @param PRINT_REQUEST true if print request required
	 * @param request PSIS sign request
	 * @param SAVE_REQUEST true if save request required
	 * @param OUT_DIR directory where to save the request
	 * @param SIGN_REQUEST name of the file where to save the request 
	 * @throws IOException
	 */
	protected void printSaveRequest(boolean PRINT_REQUEST, SignRequestDocument request,
			boolean SAVE_REQUEST, String OUT_DIR, String SIGN_REQUEST) throws IOException{
		//printing request
		if (PRINT_REQUEST){
			this.log.debug("Sign request:\n");
			Utils.printXmlObject(request);
		}
		//save request to file
		if (SAVE_REQUEST==true){
			this.log.debug("Saving request to "+OUT_DIR+SIGN_REQUEST);
			request.save(new File(OUT_DIR+SIGN_REQUEST));
		}
	}
	
	/**
	 * Logs and saves PSIS sign response to a file.
	 * @param PRINT_RESPONSE true if print response required
	 * @param response PSIS sign response
	 * @param SAVE_RESPONSE true if save response required
	 * @param OUT_DIR directory where to save the response
	 * @param SIGN_RESPONSE name of the file where to save the response
	 * @throws IOException
	 */
	protected void printSaveResponse(boolean PRINT_RESPONSE, SignResponseDocument response,
			boolean SAVE_RESPONSE, String OUT_DIR, String SIGN_RESPONSE) throws IOException{
		if(isResultWellFormed(response))
			showResult(response.getSignResponse().getResult());
		//printing response
		if (PRINT_RESPONSE){
			this.log.debug("Sign response:\n");
			Utils.printXmlObject(response);
		}
		//save response to file
		if (SAVE_RESPONSE==true){
			this.log.debug("Saving response to "+OUT_DIR+SIGN_RESPONSE);
			response.save(new File(OUT_DIR+SIGN_RESPONSE));
		}
	}

}
