package net.catcert.iarxiu.client.test;

import java.io.File;
import java.io.IOException;

import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import net.catcert.iarxiu.client.proxy.ProxyClient;
import net.catcert.iarxiu.client.test.utils.Utils;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.xml.sax.SAXException;

import com.hp.iarxiu.core.schemas.x20.ingest.ContentTypeHandlingType;
import com.hp.iarxiu.core.schemas.x20.ingest.GetOfflineIngestStatusRequestDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.GetOfflineIngestStatusResponseDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.GetUploadTicketRequestDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.GetUploadTicketResponseDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.OfflineIngestInfoType;
import com.hp.iarxiu.core.schemas.x20.ingest.OfflineUploadIngestRequestDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.OfflineUploadIngestResponseDocument;
import com.hp.iarxiu.core.schemas.x20.ingest.GetOfflineIngestStatusRequestDocument.GetOfflineIngestStatusRequest;
import com.hp.iarxiu.core.schemas.x20.ingest.GetUploadTicketResponseDocument.GetUploadTicketResponse;
import com.hp.iarxiu.core.schemas.x20.ingest.OfflineUploadIngestRequestDocument.OfflineUploadIngestRequest;


/**
 * Classe d'exemple de petici per realitzar un ingrs per upload. En un ingrs 
 * upload es puja el fitxer mets i els fitxers de l'expedient a una url. 
 * En aquesta modalitat els fitxers sempre s'han de pujar descomprimits, si 
 * es pugen comprimits no es descomprimiran, s a dir, es tractar igual un 
 * fitxer doc o pdf que un zip. 
 * 
 * Un cop s'han pujat tant el fitxer com els fitxers de l'expedient es pot fer 
 * la petici d'ingrs.  
 * 
 * A grans trets seguiem els segent passos: 
 * 1. Obtenci d'un tiquet d'upload. 
 * 2. Validaci del fitxer mets de l'expedient. (opcional)
 * 3. Upload del mets i dels fitxers de l'expedient. En el present exemple, 
 * es pugen els fitxers d'un directori concret, per no tenen perqu estar al 
 * mateix directori necessriament. 
 * 4. Obtenci d'un tiquet d'ingrs offline. 
 * 5. Consulta del resultat de l'ingrs offline a partir del tiquet d'ingrs.
 * 
 */
public class GetUploadTicketTest {

	private static final String UPLOAD_SERVLET_PARAM_TICKET = "ticket";

	public static void main(String[] args) throws Exception {
		
		/*
		 * 1. Obtenci del tiquet d'upload.
		 */
		
		// Crrega de l'application context i recuperaci de parmetres
		ApplicationContext context = new ClassPathXmlApplicationContext("applicationContext.xml");
		ProxyClient proxy = (ProxyClient) context.getBean("proxyClient");
		String uploadUrl = (String) context.getBean("uploadUrl");
		
		// Petici del tiquet d'upload
		GetUploadTicketRequestDocument requestDocument = GetUploadTicketRequestDocument.Factory.newInstance();
		requestDocument.addNewGetUploadTicketRequest();
		Utils.printXmlObject(requestDocument); // pintem la petici
		requestDocument.save(new File("samples/GetUploadTicketRequest.xml")); // guardem la request en un fitxer
		
		// Enviament de la petici
		GetUploadTicketResponseDocument responseDocument = (GetUploadTicketResponseDocument)proxy.send(requestDocument);
		Utils.printXmlObject(responseDocument);	// pintem la resposta
		responseDocument.save(new File("samples/GetUploadTicketResponse.xml")); // guardem la response en un fitxer
		
		// De la response ens interessa el ticket d'upload
		GetUploadTicketResponse response = responseDocument.getGetUploadTicketResponse();
		String uploadTicket = response.getTicket();
		
		
		/*
		 * 2. Validaci del fitxer mets de l'expedient
		 */
		String metsFilePath = "resources/upload/mets.xml";
		
		// Verifiquem la validesa de la estructura del fitxer mets
		File sourceLocation = new File(metsFilePath);
		try {
			// 1. Lookup a factory for the W3C XML Schema language
	        SchemaFactory factory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
	        
	        // 2. Compile the schema. 
	        // Here the schema is loaded from a java.io.File, but you could use 
	        // a java.net.URL or a javax.xml.transform.Source instead.
	        File schemaLocation = new File("resources/xsd/mets.xsd");
	        Schema schema = factory.newSchema(schemaLocation);
	    
	        // 3. Get a validator from the schema.
	        Validator validator = schema.newValidator();
	        
	        // 4. Parse the document you want to check.
	        Source source = new StreamSource(sourceLocation);
	        
	        // 5. Check the document
            validator.validate(source);
            System.out.println(sourceLocation.toString() + "El fitxer mets s vlid.");
        }
        catch (SAXException ex) {
            System.err.println(sourceLocation.toString() + " no s vlid perqu " + ex.getMessage());
        }catch (IOException ex) {
            System.err.println(sourceLocation.toString() + " no s vlid perqu " + ex.getMessage());
        }
        
        
        /*
         * 3. Upload del fitxer mets.xml i els fitxers de l'expedient
         */
		String certPath = "resources/upload";
		File resourcesFile = new File(certPath);
		String[] certList = resourcesFile.list();
		
		boolean uploaded = upload(uploadUrl, uploadTicket, metsFilePath, certList, certPath);
		if(!uploaded){
			System.out.println("S'ha produit un error durant la crrega del/s fitxer/s.");
		} else{
			
			/*
			 * 4. Si tots els fitxers s'han pujat correctament, fem la 
			 * sollicitud d'ingrs. L'aplicaci ens retornar el tiquet 
			 * identificatiu de la petici d'ingrs. 
			 */
			System.out.println("\nFitxer/s carregat/s correctament a iArxiu.");
			System.out.println("Sollicitant ingrs offline...");
			
			// Petici d'ingrs
			OfflineUploadIngestRequestDocument offlineUploadIngestRequestDocument = OfflineUploadIngestRequestDocument.Factory.newInstance();
			OfflineUploadIngestRequest offlineUploadIngestRequest = offlineUploadIngestRequestDocument.addNewOfflineUploadIngestRequest();
			offlineUploadIngestRequest.setUploadTicket(uploadTicket);
			
			// Preservaci d'evidncia:
			// 		true -> preservaci d'evidncia i de continguts
			// 		false -> noms preservaci dels continguts
			offlineUploadIngestRequest.setPreservation(false);  // Con valor "false" algunas excepciones no se lanzan
			//offlineUploadIngestRequest.setPreservation(true);

			// Tipus de detecci de contingut:
			//		replaceWithIntrospection -> 	Si s'identifica un format i s diferent a l'indicat pel client,
			//							   			aleshores es substitueix pel detectat.
			//		completeWithIntrospection -> 	Si s'identifica un format i s diferent a l'indicat pel client,
			//   						   			noms s'informa el PREMIS amb les dades obtingudes amb DROID. Quan 
			//   						   			no hi ha coincidncia, es respecten les dades que posa el client.
			//		checkAndReject ->			   	Si s'identifica un format i s diferent a l'indicat pel client,
			//							   			aleshores es rebutja el paquet.
			offlineUploadIngestRequest.setContentTypeHandling(ContentTypeHandlingType.COMPLETE_WITH_INTROSPECTION);
			
			// Pintem la petici d'ingrs 
			Utils.printXmlObject(offlineUploadIngestRequestDocument);
			
			// Enviament de la petici d'ingrs 
			OfflineUploadIngestResponseDocument offlineUploadIngestResponseDocument = 
				(OfflineUploadIngestResponseDocument)proxy.send(offlineUploadIngestRequestDocument);
			
			// Pintem la resposta a la petici d'ingrs
			Utils.printXmlObject(offlineUploadIngestResponseDocument);
		
			// Capturem el ticket d'ingrs
			String ingestTicket = offlineUploadIngestResponseDocument.getOfflineUploadIngestResponse();
			
			
			/*
			 * 5. Un cop obtingut el tiquet de la petici d'ingrs anem consultant  
			 * a l'aplicaci per l'estat de l'ingrs. Anirem fent la consulta 
			 * fins que l'aplicaci ens digui que ja no l'est processant.
			 */
			System.out.println("\nSollicitem estat d'ingrs offline mentres estigui en procs...");
			
			OfflineIngestInfoType.Status.Enum status = OfflineIngestInfoType.Status.IN_PROCESS;
			while(status.equals(OfflineIngestInfoType.Status.IN_PROCESS)){
			
				// Esperem 10 segons entre consulta i consulta
				Thread.sleep(10000);

				// Preparem la petici d'estat de l'ingrs
				GetOfflineIngestStatusRequestDocument getOfflineIngestStatusRequestDocument = 
					GetOfflineIngestStatusRequestDocument.Factory.newInstance();
				GetOfflineIngestStatusRequest getOfflineIngestStatusRequest = 
					getOfflineIngestStatusRequestDocument.addNewGetOfflineIngestStatusRequest();
				getOfflineIngestStatusRequest.setStringValue(ingestTicket);
				getOfflineIngestStatusRequest.setSignatureValidationDetails(true);
				
				getOfflineIngestStatusRequestDocument.setGetOfflineIngestStatusRequest(getOfflineIngestStatusRequest);
				
				// Pintem la petici sobre l'estat de l'ingrs
				Utils.printXmlObject(getOfflineIngestStatusRequestDocument);
				
				// Enviament de la petici sobre l'estat de l'ingrs
				GetOfflineIngestStatusResponseDocument getOfflineIngestStatusResponseDocument = (GetOfflineIngestStatusResponseDocument)proxy.send(getOfflineIngestStatusRequestDocument);
				
				// Pintem la resposta
				Utils.printXmlObject(getOfflineIngestStatusResponseDocument);
				
				// Capturem l'estat de l'ingrs de la resposta
				OfflineIngestInfoType statusInfo = 
					getOfflineIngestStatusResponseDocument.getGetOfflineIngestStatusResponse().getOfflineIngestInfo();
				status = statusInfo.getStatus(); // si deixa d'estar en procs es trenca el bucle
			}
		}
	}

	
	/**
	 * Upload del fitxer METS i dels fitxers adjunts al servidor.
	 * 
	 * @param uploadUrl url a la qual pujarem els fitxers 
	 * @param uploadTicket tiquet identificatiu de l'upload
	 * @param metsFilePath path absolut del fitxer METS
	 * @param attachedFilesPaths paths absoluts dels fitxers adjuntats
	 * @return true -> els fitxers s'han pujat correctament al servidor; false -> hi ha hagut algun problema
	 */
	private static boolean upload(String uploadUrl, String uploadTicket, String metsFilePath, String[] attachedFilesPaths, String certPath){
	
		try{
			File metsFile = new File(metsFilePath);
			HttpClient httpClient = new HttpClient();
			PostMethod post = new PostMethod(uploadUrl);
			
			// Identificador del tiquet d'upload
			Part ticketPart = new StringPart(UPLOAD_SERVLET_PARAM_TICKET, uploadTicket);
			
			// Arxiu METS
			Part metsPart = new FilePart("mets", metsFile);
			
			// Arxius de l'expedient
			Part[] filesParts = null;
			if(attachedFilesPaths!=null && attachedFilesPaths.length>0){
				filesParts = new Part[attachedFilesPaths.length];
				int i=0;
				while(i< attachedFilesPaths.length){
					File file = new File (certPath+"\\"+attachedFilesPaths[i].toString());
					Part filePart = new FilePart(attachedFilesPaths[i].toString(),file);
					filesParts[i] = filePart;
					i++;
				}
			}

			Part[] parts = null;
			if(filesParts==null){
				parts = new Part[]{ticketPart,metsPart};
			}
			else{
				parts = new Part[2+filesParts.length];
				parts[0] = ticketPart;
				parts[1] = metsPart;
				for(int i=0;i<filesParts.length;i++){
					parts[i+2] = filesParts[i];
				}
			}
			
			post.setRequestEntity( new MultipartRequestEntity(parts,post.getParams()) );
			
			int result = httpClient.executeMethod(post);
			
			return result==200; // true si la petici http retorna un codi 200
		
		}catch(Exception e){
			e.printStackTrace();
			return false;
		}
	}

}
